//*********************************************************************
//
//	Various math constants and routines
//
//*********************************************************************
#ifndef Math2_H
#define Math2_H

#include <math.h>
#include <mex.h>
#include <matrix.h>
#include <float.h>

#ifndef PI
#define PI 3.14159265358979
#endif
#ifndef TWOPI
#define TWOPI 6.28318530717959
#endif
#ifndef SQRT2PI
#define SQRT2PI 2.5066282746310002
#endif
#ifndef MACHEPS
#define MACHEPS 2.22044604925031e-16
#endif

inline double sqr(double x) { return x*x; }

inline double StdNormPdf(double x) {return exp(-x*x/2.)/SQRT2PI;}

inline double NormPdf(double x, double mu=0, double sd=1) {
	return (1/sd)*StdNormPdf((x-mu)/sd);
}

double normcdf2(const double, const double mu=0, const double sigma=1);
double norminv2(const double, const double mu=0, const double sigma=1);
inline double CDFN(double mu, double sd, double a, double b) {
	return normcdf2(b,mu,sd)-normcdf2(a,mu,sd);
}


#define PROBNLIMIT 8
double RandNormT(double mu, double sd, 
				 double Lower, double Upper, 
				 double randu,
				 double EPS_Inside=10.)
{
	double
		zLower = (Lower-mu)/sd,
		zUpper = (Upper-mu)/sd,
		LowerPlus = Lower+EPS_Inside*MACHEPS,
		UpperMinus = Upper-EPS_Inside*MACHEPS;
	if (zLower> PROBNLIMIT && zUpper> PROBNLIMIT) return LowerPlus;
	if (zLower<-PROBNLIMIT && zUpper<-PROBNLIMIT) return UpperMinus;
	if (zLower>0 && zLower<PROBNLIMIT) zUpper=__min(zUpper,PROBNLIMIT);
	if (zUpper<0 && zUpper>-PROBNLIMIT) zLower=__max(zLower,-PROBNLIMIT);
	double
		CDFLower = normcdf2(zLower),
		CDFUpper = normcdf2(zUpper);
	if (CDFUpper-CDFLower < 100*MACHEPS) {
		if (zLower>0 && zUpper>0) return LowerPlus;
		if (zLower<0 && zUpper<0) return UpperMinus;
		return 0.;
	}
	double F=CDFLower + (CDFUpper-CDFLower)*randu;

	if (F==0 || F==1) {
		mexPrintf("\nRandNormT F=%20.15g\n",F);
		mexPrintf("mu=%20.15g\tsd=%25.15g\n",mu,sd);
		mexPrintf("Lower=%20.15g\tUpper=%20.15g\nzLower=%20.15g\tzUpper=%20.15g\n",
			Lower, Upper, zLower, zUpper);
		mexPrintf("CDFLower=%20.15g\tCDFUpper=%20.15g\trandu=%20.15g\n",
			CDFLower, CDFUpper, randu);
		if (F==1) return UpperMinus;
		else return LowerPlus;
		//mexErrMsgTxt("Terminating.");
	}
	double draw = norminv2(F, mu, sd);
	draw = __max(draw,LowerPlus);
	draw = __min(draw,UpperMinus);
	return draw;
}

//----------------------------------------------------------------------
inline double fix(const double x)
{
	//	Round toward zero.
	if (x>=0) return floor(x);
	else return ceil(x);
}

double erfcore2(const double x, const int jint)
{
//ERFCORE Core algorithm for error functions.
//   erf(x) = erfcore(x,0)
//   erfc(x) = erfcore(x,1)
//   erfcx(x) = exp(x^2)*erfc(x) = erfcore(x,2)

//  C. Moler, 2-1-91.
//   Copyright 1984-2001 The MathWorks, Inc. 
//   $Revision: 5.13 $  $Date: 2001/04/15 12:01:41 $

//   This is a translation of a FORTRAN program by W. J. Cody,
//   Argonne National Laboratory, NETLIB/SPECFUN, March 19, 1990.
//   The main computation evaluates near-minimax approximations
//   from "Rational Chebyshev approximations for the error function"
//   by W. J. Cody, Math. Comp., 1969, PP. 631-638.
//   evaluate  erf  for  |x| <= 0.46875
    double result, xbreak = 0.46875;
	if (fabs(x)<=xbreak) {
		double a[]={3.16112374387056560e00, 1.13864154151050156e02,
			3.77485237685302021e02, 3.20937758913846947e03, 1.85777706184603153e-1};
		double b[]={2.36012909523441209e01, 2.44024637934444173e02,
			1.28261652607737228e03, 2.84423683343917062e03};
		double 
			y = fabs(x),
			z = y*y,
			xnum = a[4]*z,
			xden = z;
		for (int i=0; i<3; i++) {
			xnum = (xnum + a[i]) * z;
            xden = (xden + b[i]) * z;
		}
        result = x* (xnum + a[3]) / (xden + b[3]);
		if (jint!=0) result = 1-result;
		if (jint==2) result = exp(z)*result;
	}
//   evaluate  erfc  for 0.46875 <= |x| <= 4.0
	if (fabs(x)>xbreak && fabs(x)<=4.) {
		double 
			c[] = {5.64188496988670089e-1, 8.88314979438837594e00,
				6.61191906371416295e01, 2.98635138197400131e02,
				8.81952221241769090e02, 1.71204761263407058e03,
				2.05107837782607147e03, 1.23033935479799725e03,
				2.15311535474403846e-8},
			d[] = {1.57449261107098347e01, 1.17693950891312499e02,
				5.37181101862009858e02, 1.62138957456669019e03,
				3.29079923573345963e03, 4.36261909014324716e03,
				3.43936767414372164e03, 1.23033935480374942e03},
			y = fabs(x),
			xnum = c[8]*y,
			xden = y;
		for (int i=0; i<7; i++) {
			xnum = (xnum + c[i]) * y;
			xden = (xden + d[i]) * y;
		}
		result = (xnum + c[7])/(xden+d[7]);
		if (jint!=2) {
			double
				z = fix(y*16)/16.,
				del = (y-z)*(y+z);
			result = exp(-z*z) * exp(-del) * result;
		}
	}
//   evaluate  erfc  for |x| > 4.0
	if (fabs(x)>4.) {
		double
			p[] = {3.05326634961232344e-1, 3.60344899949804439e-1,
				1.25781726111229246e-1, 1.60837851487422766e-2,
				6.58749161529837803e-4, 1.63153871373020978e-2},
			q[] = {2.56852019228982242e00, 1.87295284992346047e00,
				5.27905102951428412e-1, 6.05183413124413191e-2,
				2.33520497626869185e-3},
			y = fabs(x),
			z = 1./(y*y),
			xnum = p[5]*z,
			xden = z;
		for (int i=0; i<4; i++) {
			xnum = (xnum + p[i]) * z;
			xden = (xden + q[i]) * z;
		}
		result = z * (xnum + p[4]) / (xden + q[4]);
        result = (1/sqrt(PI) -  result) / y;
		if (jint!=2) {
			double
				z = fix(y*16)/16,
				del = (y-z)*(y+z);
			result = exp(-z*z) * exp(-del) * result;
		}
	}
//   fix up for negative argument, erf, etc.
	switch (jint)
	{
	case (0): {
		if (x>xbreak) result = (0.5-result)+0.5;
		if (x<-xbreak) result = (-.5+result)-0.5;
		}
		break;
	case (1) : {if (x<-xbreak) result = 2.-result;}
		break;
	case (2) :
		if (x<-xbreak) {
			double
				z = fix(x*16)/16,
				del = (x-z)*(x+z),
				y = exp(z*z)*exp(del);
			result = (y+y) - result;
		}
		break;
	}
	return result;
}

double normcdf2(const double x, const double mu, const double sigma)
{
	double y = 0.5 * erfcore2( - (x - mu) / (sigma * sqrt(2)),1);
	if (y>1.) y=1.;
	return y;
}

double erfinv2(const double y)
//ERFINV Inverse error function.
//   X = ERFINV(Y) is the inverse error function for each element of Y.
//   The inverse error function satisfies y = erf(x), for -1 <= y <= 1
//   and -inf <= x <= inf.
//
//   See also ERF, ERFC, ERFCX, ERFCINV.

//   Copyright 1984-2001 The MathWorks, Inc. 
//   $Revision: 5.14 $  $Date: 2001/04/15 12:01:40 $
{
	double x=0;
	//	Coefficients in rational approximations.
	double
		a[]={ 0.886226899, -1.645349621,  0.914624893, -0.140543331},
		b[]={-2.118377725,  1.442710462, -0.329097515,  0.012229801},
		c[]={-1.970840454, -1.624906493,  3.429567803,  1.641345311},
		d[]={ 3.543889200,  1.637067800};

// Central range
	double y0 = .7;
	if (fabs(y)<=y0) {
		double z=y*y;
		x = y * (((a[3]*z+a[2])*z+a[1])*z+a[0]) / ((((b[3]*z+b[2])*z+b[1])*z+b[0])*z+1);
	}
	if (y0<y && y<1) {
		double z = sqrt(-log((1-y)/2));
		x = (((c[3]*z+c[2])*z+c[1])*z+c[0]) / ((d[1]*z+d[0])*z+1);
	}
	if ( (-y0)>y && y>-1) {
		double z = sqrt(-log((1+y)/2));
		x = -(((c[3]*z+c[2])*z+c[1])*z+c[0]) / ((d[1]*z+d[0])*z+1);
	}
//	The relative error of the approximation has absolute value less
//	than 8.9e-7.  One iteration of Halley's rational method (third
//	order) gives full machine precision.

//	Newton's method: new x = x - f/f'
//	Halley's method: new x = x - 1/(f'/f - (f"/f')/2)
//	This function: f = erf(x) - y, f' = 2/sqrt(pi)*exp(-x^2), f" = -2*x*f'

	// Newton's correction
	double u = (erfcore2(x,0) - y) / (2/sqrt(PI) * exp(-x*x));

	// Halley's step
	x = x - u/(1+x*u);

	// Exceptional cases
	//if (y==-1) x = -Inf;
	//if (y==1) x = Inf;
	//if (fabs(y)>1) x = NaN;
	//if (y==NaN) x = NaN;
	return x;
}

double norminv2(const double p, const double mu, const double sigma)
{
    return  sqrt(2) * sigma*erfinv2(2 * p - 1) + mu;
}


#endif